<?php
header('Content-Type: application/json');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: Content-Type");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$bookingsFile = __DIR__ . '/../data/bookings.json';
$settingsFile = __DIR__ . '/../data/settings.json';

function read_json($file) {
    if (!file_exists($file)) return [];
    $data = file_get_contents($file);
    $json = json_decode($data, true);
    return $json ?: [];
}

function write_json($file, $data) {
    $json = json_encode($data, JSON_PRETTY_PRINT);
    file_put_contents($file, $json);
}

function read_settings($file) {
    if (!file_exists($file)) return [];
    $data = file_get_contents($file);
    $json = json_decode($data, true);
    return $json ?: [];
}

function send_booking_emails($booking) {
    $settings = read_settings(__DIR__ . '/../data/settings.json');
    $adminEmail = $settings['admin_email'] ?? null;
    if (!$adminEmail) return;

    $subjectAdmin = "New Booking #" . $booking['id'];
    $bodyAdmin = "New booking received:\n"
        . "Pickup: " . $booking['pickup_location'] . "\n"
        . "Dropoff: " . $booking['dropoff_location'] . "\n"
        . "Date/Time: " . $booking['pickup_date'] . " " . $booking['pickup_time'] . "\n"
        . "Customer: " . $booking['customer']['first_name'] . " " . $booking['customer']['last_name'] . " (" . $booking['customer']['email'] . ")\n"
        . "Total: " . $booking['price_total'];

    @mail($adminEmail, $subjectAdmin, $bodyAdmin);

    $customerEmail = $booking['customer']['email'] ?? null;
    if ($customerEmail) {
        $subjectCust = "Your Booking Confirmation #" . $booking['id'];
        $bodyCust = "Thank you for your booking.\n\n"
            . "Pickup: " . $booking['pickup_location'] . "\n"
            . "Dropoff: " . $booking['dropoff_location'] . "\n"
            . "Date/Time: " . $booking['pickup_date'] . " " . $booking['pickup_time'] . "\n"
            . "Vehicle: " . $booking['vehicle_name'] . "\n"
            . "Total: " . $booking['price_total'] . "\n\n"
            . "Payment method: Pay on pickup.";
        @mail($customerEmail, $subjectCust, $bodyCust);
    }
}

// LIST BOOKINGS
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $bookings = read_json($bookingsFile);
    echo json_encode($bookings);
    exit;
}

// CREATE OR UPDATE BOOKING
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);

    if (!$input) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid JSON']);
        exit;
    }

    $bookings = read_json($bookingsFile);

    // UPDATE (status, driver, etc.)
    if (isset($input['action']) && $input['action'] === 'update' && isset($input['id'])) {
        foreach ($bookings as &$bk) {
            if ($bk['id'] == $input['id']) {
                if (isset($input['status'])) $bk['status'] = $input['status'];
                if (isset($input['driver_id'])) $bk['driver_id'] = $input['driver_id'];
                break;
            }
        }
        write_json($bookingsFile, $bookings);
        echo json_encode(['success' => true, 'bookings' => $bookings]);
        exit;
    }

    // CREATE
    $newId = count($bookings) > 0 ? end($bookings)['id'] + 1 : 1;
    $now = date('Y-m-d H:i:s');

    $booking = [
        'id' => $newId,
        'created_at' => $now,
        'service_type' => $input['service_type'] ?? null,
        'pickup_date' => $input['pickup_date'] ?? null,
        'pickup_time' => $input['pickup_time'] ?? null,
        'pickup_location' => $input['pickup_location'] ?? null,
        'dropoff_location' => $input['dropoff_location'] ?? null,
        'passengers' => (int)($input['passengers'] ?? 1),
        'luggage' => (int)($input['luggage'] ?? 0),
        'child_seats' => $input['child_seats'] ?? [],
        'vehicle_id' => (int)($input['vehicle_id'] ?? 0),
        'vehicle_name' => $input['vehicle_name'] ?? '',
        'distance_miles' => (float)($input['distance_miles'] ?? 0),
        'price_total' => (float)($input['price_total'] ?? 0),
        'promo_code' => $input['promo_code'] ?? null,
        'payment_method' => 'pay_on_pickup',
        'driver_id' => $input['driver_id'] ?? null,
        'customer' => [
            'first_name' => $input['first_name'] ?? '',
            'last_name' => $input['last_name'] ?? '',
            'email' => $input['email'] ?? '',
            'phone' => $input['phone'] ?? ''
        ],
        'status' => 'pending'
    ];

    $bookings[] = $booking;
    write_json($bookingsFile, $bookings);

    // email notifications
    send_booking_emails($booking);

    echo json_encode(['success' => true, 'booking' => $booking]);
    exit;
}

http_response_code(405);
echo json_encode(['error' => 'Method not allowed']);
