
const apiBase = "../api";
const dataBase = "../data";

const bookingState = {
    serviceTypes: [],
    fleet: [],
    promoCodes: [],
    selectedVehicle: null,
    childSeats: [],
    appliedPromo: null
};

/* UTILITIES */
async function fetchJSON(url) {
    const res = await fetch(url);
    if (!res.ok) throw new Error("Network error");
    return res.json();
}

async function postJSON(url, data) {
    const res = await fetch(url, {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(data)
    });
    if (!res.ok) throw new Error("Network error");
    return res.json();
}

/* PAGE DETECTION */
document.addEventListener("DOMContentLoaded", () => {
    const bookingPage = document.getElementById("step-1");
    const adminPage = document.querySelector(".admin-layout-full");

    if (bookingPage) initBookingPage();
    if (adminPage) initAdminPage();
});

/* ---------- BOOKING FRONTEND ---------- */
async function initBookingPage() {
    try {
        bookingState.serviceTypes = await fetchJSON(dataBase + "/service_types.json");
        const select = document.getElementById("serviceType");
        bookingState.serviceTypes.forEach(st => {
            const opt = document.createElement("option");
            opt.value = st.code;
            opt.textContent = st.name;
            select.appendChild(opt);
        });
    } catch (e) { console.error(e); }

    try {
        bookingState.fleet = await fetchJSON(apiBase + "/fleet.php");
    } catch (e) { console.error(e); }

    try {
        bookingState.promoCodes = await fetchJSON(apiBase + "/promo.php");
    } catch (e) { console.error(e); }

    document.getElementById("addChildSeatBtn").addEventListener("click", () => {
        const type = document.getElementById("childSeatType").value;
        const qty = parseInt(document.getElementById("childSeatQty").value || "0", 10);
        if (!type || qty <= 0) return;
        bookingState.childSeats.push({ type, qty });
        renderChildSeats();
    });

    document.getElementById("toStep2").addEventListener("click", goToStep2);
    document.getElementById("toStep3").addEventListener("click", goToStep3);
    document.getElementById("submitBooking").addEventListener("click", submitBooking);
    document.getElementById("applyPromoBtn").addEventListener("click", applyPromoCode);

    document.querySelectorAll("button[data-prev]").forEach(btn => {
        btn.addEventListener("click", () => showStep(btn.dataset.prev));
    });
}

function renderChildSeats() {
    const list = document.getElementById("childSeatList");
    list.innerHTML = "";
    bookingState.childSeats.forEach((cs, index) => {
        const li = document.createElement("li");
        li.className = "pill";
        li.textContent = cs.type + " x " + cs.qty;
        li.title = "Click to remove";
        li.addEventListener("click", () => {
            bookingState.childSeats.splice(index, 1);
            renderChildSeats();
        });
        list.appendChild(li);
    });
}

function showStep(step) {
    document.querySelectorAll(".step-panel").forEach(p => p.classList.remove("active"));
    document.querySelectorAll(".step").forEach(s => s.classList.remove("active"));
    document.getElementById("step-" + step).classList.add("active");
    document.querySelector('.step[data-step="' + step + '"]').classList.add("active");
}

function getStep1Data() {
    return {
        service_type: document.getElementById("serviceType").value,
        pickup_date: document.getElementById("pickupDate").value,
        pickup_time: document.getElementById("pickupTime").value,
        pickup_location: document.getElementById("pickupLocation").value,
        dropoff_location: document.getElementById("dropoffLocation").value,
        passengers: parseInt(document.getElementById("passengers").value || "1", 10),
        luggage: parseInt(document.getElementById("luggage").value || "0", 10),
        child_seats: bookingState.childSeats,
        distance_miles: parseFloat(document.getElementById("distanceMiles").value || "0")
    };
}

function goToStep2() {
    const s1 = getStep1Data();
    if (!s1.service_type || !s1.pickup_date || !s1.pickup_time || !s1.pickup_location || !s1.dropoff_location) {
        alert("Please fill in all required fields in Step 1.");
        return;
    }
    renderVehicleList(s1);
    showStep(2);
}

function findServiceType(code) {
    return bookingState.serviceTypes.find(x => x.code === code);
}

function findPromo(code) {
    if (!code) return null;
    const c = code.trim().toUpperCase();
    return bookingState.promoCodes.find(p => p.code.toUpperCase() === c && p.active);
}

function calculatePrice(vehicle, distanceMiles, serviceTypeCode, promoCode) {
    const base = parseFloat(vehicle.base_price || 0);
    const perMile = parseFloat(vehicle.price_per_mile || 0);
    const distanceCost = distanceMiles * perMile;
    let total = base + distanceCost;

    const st = findServiceType(serviceTypeCode);
    if (st && st.discount) {
        total = total - total * parseFloat(st.discount);
    }

    const promo = findPromo(promoCode);
    if (promo) {
        if (promo.type === "percent") {
            total = total - total * (promo.value / 100);
        } else if (promo.type === "fixed") {
            total = total - promo.value;
        }
    }
    return Math.max(total, 0);
}

function renderVehicleList(s1) {
    const container = document.getElementById("vehicleList");
    container.innerHTML = "";

    const matching = bookingState.fleet.filter(v =>
        v.active &&
        v.max_passengers >= s1.passengers &&
        v.max_luggage >= s1.luggage
    );

    if (matching.length === 0) {
        container.innerHTML = "<p>No vehicles available for this passenger/luggage combination.</p>";
        return;
    }

    matching.forEach(v => {
        const price = calculatePrice(
            v,
            s1.distance_miles,
            s1.service_type,
            bookingState.appliedPromo ? bookingState.appliedPromo.code : null
        );

        const card = document.createElement("div");
        card.className = "vehicle-card";
        card.innerHTML = `
            <img src="${v.image}" alt="${v.name}">
            <h3>${v.name}</h3>
            <p>Passengers: ${v.max_passengers} | Luggage: ${v.max_luggage}</p>
            <p><strong>$${price.toFixed(2)}</strong></p>
            <button class="btn-primary">Select</button>
        `;

        card.querySelector("button").addEventListener("click", () => {
            bookingState.selectedVehicle = { ...v, price };
            document.querySelectorAll(".vehicle-card").forEach(c => c.classList.remove("selected"));
            card.classList.add("selected");
            document.getElementById("toStep3").disabled = false;
        });

        container.appendChild(card);
    });
}

function goToStep3() {
    if (!bookingState.selectedVehicle) {
        alert("Please select a vehicle.");
        return;
    }
    const s1 = getStep1Data();
    const v = bookingState.selectedVehicle;
    const promo = bookingState.appliedPromo;
    const summary = document.getElementById("bookingSummary");
    summary.innerHTML = `
        <h3>Booking Summary</h3>
        <p><strong>Service:</strong> ${s1.service_type}</p>
        <p><strong>Pickup:</strong> ${s1.pickup_location} on ${s1.pickup_date} at ${s1.pickup_time}</p>
        <p><strong>Dropoff:</strong> ${s1.dropoff_location}</p>
        <p><strong>Passengers:</strong> ${s1.passengers}, <strong>Luggage:</strong> ${s1.luggage}</p>
        <p><strong>Vehicle:</strong> ${v.name}</p>
        <p><strong>Estimated Distance:</strong> ${s1.distance_miles.toFixed(1)} mi</p>
        ${promo ? `<p><strong>Promo applied:</strong> ${promo.code}</p>` : ""}
        <p><strong>Total:</strong> $${v.price.toFixed(2)}</p>
    `;
    showStep(3);
}

function applyPromoCode() {
    const codeInput = document.getElementById("promoCode").value;
    const msg = document.getElementById("promoMessage");
    msg.textContent = "";
    msg.className = "message";
    const promo = findPromo(codeInput);
    if (!promo) {
        bookingState.appliedPromo = null;
        msg.classList.add("error");
        msg.textContent = "Promo code not valid.";
        return;
    }
    bookingState.appliedPromo = promo;
    msg.classList.add("success");
    msg.textContent = "Promo code applied: " + promo.code;
}

async function submitBooking() {
    const s1 = getStep1Data();
    const v = bookingState.selectedVehicle;
    const msgEl = document.getElementById("bookingMessage");
    msgEl.textContent = "";
    msgEl.className = "message";

    const payload = {
        ...s1,
        vehicle_id: v.id,
        vehicle_name: v.name,
        price_total: v.price,
        promo_code: bookingState.appliedPromo ? bookingState.appliedPromo.code : null,
        first_name: document.getElementById("firstName").value,
        last_name: document.getElementById("lastName").value,
        email: document.getElementById("email").value,
        phone: document.getElementById("phone").value
    };

    if (!payload.first_name || !payload.last_name || !payload.email || !payload.phone) {
        alert("Please fill in your contact details.");
        return;
    }

    try {
        const result = await postJSON(apiBase + "/booking.php", payload);
        if (result.success) {
            msgEl.classList.add("success");
            msgEl.textContent = "Booking confirmed! Your ID is " + result.booking.id + ". A confirmation email has been sent.";
        } else {
            msgEl.classList.add("error");
            msgEl.textContent = "There was an error saving your booking.";
        }
    } catch (e) {
        console.error(e);
        msgEl.classList.add("error");
        msgEl.textContent = "Network error. Please try again.";
    }
}

/* ---------- ADMIN FRONTEND ---------- */

async function initAdminPage() {
    // nav switching
    document.querySelectorAll(".nav-item").forEach(btn => {
        btn.addEventListener("click", () => {
            document.querySelectorAll(".nav-item").forEach(b => b.classList.remove("active"));
            document.querySelectorAll(".admin-view").forEach(v => v.classList.remove("active"));
            btn.classList.add("active");
            document.getElementById("view-" + btn.dataset.view).classList.add("active");
        });
    });

    await Promise.all([
        loadAdminBookings(),
        loadAdminFleet(),
        loadAdminDrivers(),
        loadAdminPromo()
    ]);
    computeDashboardStats();

    const searchInput = document.getElementById("bookingSearch");
    const statusFilter = document.getElementById("bookingStatusFilter");
    if (searchInput) searchInput.addEventListener("input", filterBookingsTable);
    if (statusFilter) statusFilter.addEventListener("change", filterBookingsTable);

    const addVehicleBtn = document.getElementById("addVehicleBtn");
    const saveFleetBtn = document.getElementById("saveFleetBtn");
    if (addVehicleBtn) addVehicleBtn.addEventListener("click", addVehicleRow);
    if (saveFleetBtn) saveFleetBtn.addEventListener("click", saveAdminFleet);

    const addDriverBtn = document.getElementById("addDriverBtn");
    const saveDriversBtn = document.getElementById("saveDriversBtn");
    if (addDriverBtn) addDriverBtn.addEventListener("click", addDriverRow);
    if (saveDriversBtn) saveDriversBtn.addEventListener("click", saveAdminDrivers);
}

/* BOOKINGS */
let adminBookings = [];

async function loadAdminBookings() {
    try {
        adminBookings = await fetchJSON(apiBase + "/booking.php");
        renderBookingsTable(adminBookings);
    } catch (e) { console.error(e); }
}

function renderBookingsTable(data) {
    const tbody = document.querySelector("#bookingsTable tbody");
    if (!tbody) return;
    tbody.innerHTML = "";
    data.forEach(b => {
        const tr = document.createElement("tr");
        const status = b.status || "pending";
        tr.innerHTML = `
            <td>${b.id}</td>
            <td>${b.created_at}</td>
            <td>${b.pickup_location}</td>
            <td>${b.dropoff_location}</td>
            <td>${b.pickup_date} ${b.pickup_time}</td>
            <td>${b.vehicle_name}</td>
            <td>${b.customer.first_name} ${b.customer.last_name}</td>
            <td><span class="badge ${status}">${status}</span></td>
            <td>$${parseFloat(b.price_total || 0).toFixed(2)}</td>
        `;
        tbody.appendChild(tr);
    });
}

function filterBookingsTable() {
    const q = (document.getElementById("bookingSearch").value || "").toLowerCase();
    const status = document.getElementById("bookingStatusFilter").value;
    const filtered = adminBookings.filter(b => {
        const hay = (b.pickup_location + " " + b.dropoff_location + " " + b.vehicle_name + " " +
            b.customer.first_name + " " + b.customer.last_name).toLowerCase();
        const matchesText = hay.includes(q);
        const matchesStatus = !status || (b.status || "pending") === status;
        return matchesText && matchesStatus;
    });
    renderBookingsTable(filtered);
}

function computeDashboardStats() {
    const todayStr = new Date().toISOString().slice(0,10);
    let total = adminBookings.length;
    let today = 0;
    let upcoming = 0;
    let revenue = 0;
    adminBookings.forEach(b => {
        if (b.pickup_date === todayStr) today++;
        if (b.status === "confirmed" || b.status === "pending") upcoming++;
        revenue += parseFloat(b.price_total || 0);
    });
    const fmt = v => "$" + v.toFixed(2);
    const elT = document.getElementById("stat-today");
    const elU = document.getElementById("stat-upcoming");
    const elTot = document.getElementById("stat-total");
    const elRev = document.getElementById("stat-revenue");
    if (elT) elT.textContent = today;
    if (elU) elU.textContent = upcoming;
    if (elTot) elTot.textContent = total;
    if (elRev) elRev.textContent = fmt(revenue);
}

/* FLEET */
let adminFleet = [];

async function loadAdminFleet() {
    try {
        adminFleet = await fetchJSON(apiBase + "/fleet.php");
        renderAdminFleet();
    } catch (e) { console.error(e); }
}

function addVehicleRow() {
    const maxId = adminFleet.reduce((m, v) => Math.max(m, v.id), 0);
    adminFleet.push({
        id: maxId + 1,
        name: "New Vehicle",
        type: "sedan",
        max_passengers: 3,
        max_luggage: 3,
        base_price: 0,
        price_per_mile: 0,
        image: "",
        active: true
    });
    renderAdminFleet();
}

function renderAdminFleet() {
    const tbody = document.querySelector("#fleetTable tbody");
    if (!tbody) return;
    tbody.innerHTML = "";
    adminFleet.forEach((v, index) => {
        const tr = document.createElement("tr");
        tr.innerHTML = `
            <td>${v.id}</td>
            <td>
                ${v.image ? `<img src="../public/${v.image}" onerror="this.src='${v.image}'" width="70">` : ""}
                <input type="file" class="upload-img">
            </td>
            <td><input value="${v.name}"></td>
            <td><input value="${v.type}"></td>
            <td><input type="number" min="1" value="${v.max_passengers}"></td>
            <td><input type="number" min="0" value="${v.max_luggage}"></td>
            <td><input type="number" min="0" step="0.01" value="${v.base_price}"></td>
            <td><input type="number" min="0" step="0.01" value="${v.price_per_mile}"></td>
            <td><input type="checkbox" ${v.active ? "checked" : ""}></td>
            <td><button class="btn-secondary btn-small">Delete</button></td>
        `;
        const inputs = tr.querySelectorAll("td input");
        // file input is inputs[0]
        const fileInput = inputs[0];
        const nameInput = inputs[1];
        const typeInput = inputs[2];
        const paxInput = inputs[3];
        const lugInput = inputs[4];
        const baseInput = inputs[5];
        const mileInput = inputs[6];
        const activeInput = inputs[7];

        nameInput.addEventListener("input", e => adminFleet[index].name = e.target.value);
        typeInput.addEventListener("input", e => adminFleet[index].type = e.target.value);
        paxInput.addEventListener("input", e => adminFleet[index].max_passengers = parseInt(e.target.value || "0", 10));
        lugInput.addEventListener("input", e => adminFleet[index].max_luggage = parseInt(e.target.value || "0", 10));
        baseInput.addEventListener("input", e => adminFleet[index].base_price = parseFloat(e.target.value || "0"));
        mileInput.addEventListener("input", e => adminFleet[index].price_per_mile = parseFloat(e.target.value || "0"));
        activeInput.addEventListener("change", e => adminFleet[index].active = e.target.checked);

        fileInput.addEventListener("change", async e => {
            if (!e.target.files.length) return;
            const formData = new FormData();
            formData.append("file", e.target.files[0]);
            try {
                const res = await fetch(apiBase + "/upload.php", {
                    method: "POST",
                    body: formData
                });
                const result = await res.json();
                if (result.success) {
                    adminFleet[index].image = result.url;
                    renderAdminFleet();
                } else {
                    alert("Image upload failed");
                }
            } catch (err) {
                console.error(err);
                alert("Image upload error");
            }
        });

        tr.querySelector("button").addEventListener("click", () => {
            adminFleet.splice(index, 1);
            renderAdminFleet();
        });

        tbody.appendChild(tr);
    });
}

async function saveAdminFleet() {
    const msg = document.getElementById("fleetMessage");
    if (msg) {
        msg.textContent = "";
        msg.className = "message";
    }
    try {
        const result = await postJSON(apiBase + "/fleet.php", adminFleet);
        if (result.success && msg) {
            msg.classList.add("success");
            msg.textContent = "Fleet saved successfully.";
        }
    } catch (e) {
        console.error(e);
        if (msg) {
            msg.classList.add("error");
            msg.textContent = "Error saving fleet.";
        }
    }
}

/* DRIVERS */
let adminDrivers = [];

async function loadAdminDrivers() {
    try {
        adminDrivers = await fetchJSON(apiBase + "/drivers.php");
        renderAdminDrivers();
    } catch (e) { console.error(e); }
}

function renderAdminDrivers() {
    const tbody = document.querySelector("#driversTable tbody");
    if (!tbody) return;
    tbody.innerHTML = "";
    adminDrivers.forEach((d, index) => {
        const tr = document.createElement("tr");
        tr.innerHTML = `
            <td>${d.id}</td>
            <td><input value="${d.name}"></td>
            <td><input value="${d.phone}"></td>
            <td><input value="${d.email}"></td>
            <td><input type="checkbox" ${d.active ? "checked" : ""}></td>
            <td><button class="btn-secondary btn-small">Delete</button></td>
        `;
        const inputs = tr.querySelectorAll("input");
        const nameInput = inputs[0];
        const phoneInput = inputs[1];
        const emailInput = inputs[2];
        const activeInput = inputs[3];

        nameInput.addEventListener("input", e => adminDrivers[index].name = e.target.value);
        phoneInput.addEventListener("input", e => adminDrivers[index].phone = e.target.value);
        emailInput.addEventListener("input", e => adminDrivers[index].email = e.target.value);
        activeInput.addEventListener("change", e => adminDrivers[index].active = e.target.checked);

        tr.querySelector("button").addEventListener("click", () => {
            adminDrivers.splice(index, 1);
            renderAdminDrivers();
        });

        tbody.appendChild(tr);
    });
}

function addDriverRow() {
    const maxId = adminDrivers.reduce((m, d) => Math.max(m, d.id), 0);
    adminDrivers.push({
        id: maxId + 1,
        name: "New Driver",
        phone: "",
        email: "",
        notes: "",
        active: true
    });
    renderAdminDrivers();
}

async function saveAdminDrivers() {
    const msg = document.getElementById("driversMessage");
    if (msg) {
        msg.textContent = "";
        msg.className = "message";
    }
    try {
        const result = await postJSON(apiBase + "/drivers.php", adminDrivers);
        if (result.success && msg) {
            msg.classList.add("success");
            msg.textContent = "Drivers saved successfully.";
        }
    } catch (e) {
        console.error(e);
        if (msg) {
            msg.classList.add("error");
            msg.textContent = "Error saving drivers.";
        }
    }
}

/* PROMO LIST */
async function loadAdminPromo() {
    try {
        const list = await fetchJSON(apiBase + "/promo.php");
        const tbody = document.querySelector("#promoTable tbody");
        if (!tbody) return;
        tbody.innerHTML = "";
        list.forEach(p => {
            const tr = document.createElement("tr");
            tr.innerHTML = `
                <td>${p.code}</td>
                <td>${p.type}</td>
                <td>${p.type === "percent" ? p.value + "%" : "$" + p.value}</td>
                <td>${p.description}</td>
                <td>${p.active ? "Yes" : "No"}</td>
            `;
            tbody.appendChild(tr);
        });
    } catch (e) {
        console.error(e);
    }
}
